﻿/*
 *    LeapPrints
 *    http://foxycrafts.com
 *
 *    (C) 2017, Adam Murphy
 *
 */
using System;
using System.IO;
using System.IO.Pipes;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Printing;
using System.Printing;
using System.Text.RegularExpressions;
using System.Collections.Generic;

namespace LeapPrinter
{
    public struct CheckoutSlip
    {
        public string header, where, date, num, name, footer;
        private List<CKOItem> items;

        public CheckoutSlip(string h, string w, string d, string nm, string n, string f)
        {
            header = h;
            where = w;
            date = d;
            num = nm;
            name = n;
            footer = f;
            items = new List<CKOItem>();
        }

        public void reset()
        {
            header = null;
            where = null;
            date = null;
            num = null;
            name = null;
            footer = null;
            if (items != null) { items.Clear(); }
        }

        public void addItem(CKOItem item)
        {
            //make sure the list has been initialized
            if (items == null) { items = new List<CKOItem>(); }
            items.Add(item);
        }

        public List<CKOItem> getItems()
        {
            //make sure the list has been initialized
            if (items == null) { items = new List<CKOItem>(); }

            return items;
        }

        public string toString()
        {
            string slip;

            //make sure the list has been initialized
            if (items == null) { items = new List<CKOItem>(); }

            slip = header;
            slip += Environment.NewLine + Environment.NewLine + where;
            slip += Environment.NewLine + Environment.NewLine + date;
            slip += Environment.NewLine + num;
            slip += Environment.NewLine + name;
            foreach (CKOItem i in items)
            {
                slip += Environment.NewLine + Environment.NewLine + i.toGenericString();
            }
            slip += Environment.NewLine + Environment.NewLine + footer;

            return slip;
        }
    }

    public struct CKOItem
    {
        public string num, title, callno, material, due;
        int linelength;

        //create a CKOItem with no line length
        public CKOItem(string n, string t, string c, string m, string d)
        {
            num = n;
            title = t;
            callno = c;
            material = m;
            due = d;
            linelength = 0;
        }

        //create a CKOItem with a specified line length
        public CKOItem(string n, string t, string c, string m, string d, int l)
        {
            num = n;
            title = t;
            callno = c;
            material = m;
            due = d;
            linelength = l;
        }

        public void setLineLength(int length)
        {
            if (length < 1) { linelength = 0; }
            else { linelength = length; }
        }

        public string toFormattedString()
        {
            string slip = "Item: " + num + Environment.NewLine;
            slip += "Title: " + title + Environment.NewLine;
            slip += "Call no: " + callno + Environment.NewLine;
            slip += "Material: " + material + Environment.NewLine;
            slip += "Due: " + due;

            return slip;
        }

        public string toGenericString()
        {

            string slip = "";
            if (linelength < 1)
            {
                slip = num + Environment.NewLine;
                slip += title + Environment.NewLine;
                slip += callno + Environment.NewLine;
                slip += material + Environment.NewLine;
                slip += due;
            }
            else
            {

            }
            return slip;
        }

    }
    public struct TransitSlip
    {
        public string line1, line2, line3, body;

        public TransitSlip(string l1, string l2, string l3, string b)
        {
            line1 = l1;
            line2 = l2;
            line3 = l3;
            body = b;
        }
    }

    public struct HoldSlip
    {
        public string home, pickup, barcode, title, name;

        public HoldSlip(string h, string p, string b, string t, string n)
        {
            home = h;
            pickup = p;
            barcode = b;
            title = t;
            name = n;
        }

        public bool hasMiddleInit()
        {
            return (name.Length - name.IndexOf(',')) > 1;
        }

        public void reset()
        {
            home = null;
            pickup = null;
            barcode = null;
            title = null;
            name = null;
        }

        public string toString()
        {
            return home + Environment.NewLine + pickup + Environment.NewLine + barcode + Environment.NewLine + title + Environment.NewLine + name;
        }
    }

    public struct PaymentSlip
    {
        public string header, items, footer;

        public PaymentSlip(string h, string i, string f)
        {
            header = h;
            items = i;
            footer = f;
        }

        public void reset()
        {
            header = null;
            items = null;
            footer = null;
        }

        public string toString()
        {
            return header + items + footer;
        }

    }

    public struct CheckinSlip
    {
        public string header, items;

        public CheckinSlip(string h, string i)
        {
            header = h;
            items = i;
        }

        public void reset()
        {
            header = null;
            items = null;
        }

        public string toString()
        {
            items = Regex.Replace(items, "Barcode:", System.Environment.NewLine + "Barcode:");
            return header + items;
        }
    }

    public class SlipManager
    {
        //private static int numThreads = 1;
        private const string TRANSIT = "TRANSIT";
        private const string HOLD = "HOLD";
        private const string CKO = "CKO";
        private const string PAYMENT = "PAYMENT";
        private const string CKI = "CKI";
        private const string NONE = "NONE";

        private NamedPipeServerStream[] pipeServer;
        private AsyncCallback pipeAsyncCallBack;
        private Boolean[] pipeOpen;
        private IAsyncResult[] myPipeResult;
        private StreamReader streamToPrint;
        private string printedPage;

        private HoldSlip holdSlip;
        private TransitSlip transitSlip;
        private CheckoutSlip ckoSlip;
        private PaymentSlip paymentSlip;
        private CheckinSlip ckiSlip;

        private string pageType;

        private string receiptPrinter;
        private string officePrinter;

        PaperSize paperSize = new PaperSize("Roll Paper", 300, 1100);

        public SlipManager(string rprinter, string oprinter)
        {
            this.pipeServer = new NamedPipeServerStream[2];
            this.pipeAsyncCallBack = new AsyncCallback(pipeCallBack);
            this.pipeOpen = new Boolean[pipeServer.Length];
            this.myPipeResult = new IAsyncResult[pipeServer.Length];
            this.printedPage = "";
            this.holdSlip = new HoldSlip();
            this.ckoSlip = new CheckoutSlip();
            this.receiptPrinter = rprinter;
            this.officePrinter = oprinter;
        }

        public static void Main(string[] args)
        {
            SlipManager slipManager;
            if (args.Length == 2)
            {
                slipManager = new SlipManager(args[0], args[1]);
            }
            else
            {
                //ps = new PipeServer("Microsoft XPS Document Writer", "HPF3EBFE (HP ENVY 4500 series)");
                slipManager = new SlipManager("Microsoft XPS Document Writer", "Microsoft Print to PDF");
                //return;
            }
            Console.WriteLine(slipManager.receiptPrinter);

            //create the system tray icon which has the menu item "Exit"
            ContextMenu myContextMenu = new ContextMenu(new MenuItem[] {
                //new MenuItem("Reload Configuration", HandleReload),
                //new MenuItem("Reset Printer", AddressOf HandleReset),
                new MenuItem("-"),
                new MenuItem("Close Receipt Printer", new System.EventHandler(HandleExit))
        });
            NotifyIcon myNotifyIcon = new NotifyIcon();
            myNotifyIcon.Icon = new Icon("favicon.ico");
            myNotifyIcon.Text = "Receipt Printer";
            myNotifyIcon.ContextMenu = myContextMenu;
            myNotifyIcon.Visible = true;

            Console.WriteLine("\n*** Named pipe server stream with impersonation example ***\n");
            Console.WriteLine("Opening Pipes");

            //open pipes and wait for connections
            for (int i = 0; i < slipManager.pipeServer.Length; i++)
            {
                Console.WriteLine("Waiting for client connect...\n");
                slipManager.pipeServer[i] = new NamedPipeServerStream("PolReceipt", PipeDirection.In, 10, PipeTransmissionMode.Message, PipeOptions.Asynchronous);
                slipManager.myPipeResult[i] = slipManager.pipeServer[i].BeginWaitForConnection(slipManager.pipeAsyncCallBack, slipManager.pipeOpen[i]);
                slipManager.pipeOpen[i] = true;
            }
            Console.WriteLine("Beginning Run Loop");
            Application.Run();
            Console.WriteLine("Closing Pipes");
            myNotifyIcon.Visible = false;

            //close all open pipes
            for (int i = 0; i < slipManager.pipeServer.Length; i++)
            {
                slipManager.pipeOpen[i] = false;
                slipManager.pipeServer[i].Close();
            }
            //ps.streamToPrint.Close();
            Console.WriteLine("Done");

            Console.WriteLine("\nServer threads exhausted, exiting.");
        }

        public static void HandleExit(Object sender, System.EventArgs e)
        {
            Application.Exit();
        }

        private void pipeCallBack(IAsyncResult result)
        {
            Console.WriteLine("PipeCallBack: " + result.ToString());
            int i;

            for (i = 0; i < pipeServer.Length; i++)
            {
                Console.WriteLine("PipeCallBack checking result: " + i);
                if (result == myPipeResult[i])
                {
                    Console.WriteLine("PipeCallBack result found");
                    break;
                }
            }
            Console.WriteLine("PipeCallBack is pipe open: " + pipeOpen[i]);
            if (pipeOpen[i])
            {
                Console.WriteLine("PipeCallBack end connection");
                pipeServer[i].EndWaitForConnection(result);
                Console.WriteLine("PipeCallBack streamreader");
                this.streamToPrint = new StreamReader(pipeServer[i], System.Text.Encoding.ASCII);
                printReceipt();
                Console.WriteLine("PipeCallBack Disconnect");
                pipeServer[i].Disconnect();
                Console.WriteLine("PipeCallBack BeginWaitForConnection");
                myPipeResult[i] = pipeServer[i].BeginWaitForConnection(pipeAsyncCallBack, pipeOpen[i]);

            }
            Console.WriteLine("End PipeCallBack");
        }

        public void printReceipt()
        {
            try
            {

                printedPage = streamToPrint.ReadToEnd().Trim();
                Console.WriteLine("-----------------");
                Console.WriteLine(printedPage);
                Console.WriteLine("-----------------");
                //Get the printed text and replace any multiple spaces into one
                printedPage = Regex.Replace(printedPage, " {2,}", " ");
                //printedPage = Regex.Replace(printedPage, @"(\w)(\s){1,1}(\w)", "$1.$3");
                //printedPage = Regex.Replace(printedPage, @"[\t]{2,}", ".");
                Console.WriteLine(printedPage);
                Console.WriteLine("-----------------");
                printedPage = Regex.Replace(printedPage, @"(\r?\n)\1+", "$1");

                //Console.WriteLine("Before decode: " + printedPage);

                //Console.WriteLine("After decode: " + printedPage);
                //printedPage = Regex.Replace(printedPage.Trim(), " {2,}", " ");
                //Console.WriteLine(printedPage);
                /*if (printedPage.Contains("Transit")) {
                    RawPrinterHelper.SendprintedPageer(receiptPrinter, (printedPage+"\f"));
                }
                else
                {
                    RawPrinterHelper.SendprintedPageer(receiptPrinter, (printedPage + "\f"));
                }*/

                PrintDocument pd = new PrintDocument();
                //pd.PrintPage += new PrintPageEventHandler(this.pd_PrintPageStream);
                //connect the print page handler to the function that outputs the printed page
                pd.PrintPage += new PrintPageEventHandler(this.pd_PrintPageString);

                //transit slips start with "To:" and contain "IN-TRANSIT"
                if (printedPage.StartsWith("To:") && printedPage.Contains("IN-TRANSIT"))
                {
                    pageType = TRANSIT;
                    processTransit();
                    pd.PrinterSettings.PrinterName = receiptPrinter;
                    pd.DefaultPageSettings.PaperSize = paperSize;
                    pd.PrinterSettings.DefaultPageSettings.PaperSize = paperSize;
                }
                //checkout slips start with "Check Out"
                else if (printedPage.StartsWith("Check Out"))
                {
                    pageType = CKO;
                    Console.WriteLine("---Process CKO Slip---");
                    processCheckout();
                    printedPage = ckoSlip.toString();
                    pd.PrinterSettings.PrinterName = receiptPrinter;

                    pd.DefaultPageSettings.PaperSize = paperSize;
                    pd.PrinterSettings.DefaultPageSettings.PaperSize = paperSize;
                    Console.WriteLine("---End CKO Slip Setup---");
                }
                //payment slips start with "Payment Receipt"
                else if (printedPage.StartsWith("Payment Receipt"))
                {
                    pageType = PAYMENT;
                    Console.WriteLine("---Process PAYMENT Slip---");
                    processPayment();
                    printedPage = paymentSlip.toString();
                    pd.PrinterSettings.PrinterName = receiptPrinter;

                    pd.DefaultPageSettings.PaperSize = paperSize;
                    pd.PrinterSettings.DefaultPageSettings.PaperSize = paperSize;
                    Console.WriteLine("---End PAYMENT Slip Setup---");
                }
                //checkin slips start with "CHECK-IN RECEIPT"
                else if (printedPage.StartsWith("CHECK-IN RECEIPT"))
                {
                    pageType = CKI;
                    Console.WriteLine("---Process CHECK-IN Slip---");
                    processCheckin();
                    printedPage = ckiSlip.toString();
                    pd.PrinterSettings.PrinterName = receiptPrinter;

                    pd.DefaultPageSettings.PaperSize = paperSize;
                    pd.PrinterSettings.DefaultPageSettings.PaperSize = paperSize;
                    Console.WriteLine("---End CHECK-IN Slip Setup---");
                }
                //hold slips contain "Pickup By"; last just in case above slips contain "Pickup By"
                else if (printedPage.Contains("Pickup By"))
                {
                    pageType = HOLD;
                    processHold();
                    //Set the page orientation to landscape.
                    pd.DefaultPageSettings.Landscape = true;
                    pd.PrinterSettings.PrinterName = officePrinter;
                }
                //print everything else to the office printer
                else
                {
                    pageType = NONE;
                    pd.DefaultPageSettings.Landscape = true;
                    pd.PrinterSettings.PrinterName = officePrinter;
                }
                //printed list of items
                pd.Print();
                pd.Dispose();
            }
            catch (Exception ex)
            {
                throw new Exception("Exception Occured While Printing", ex);
            }

        }

        // The PrintPage event is raised for each page to be printed.
        private void pd_PrintPageStream(object sender, PrintPageEventArgs ev)
        {
            float linesPerPage = 0;
            float yPos = 0;
            int count = 0;
            float leftMargin = ev.MarginBounds.Left;
            float topMargin = ev.MarginBounds.Top;
            string line = null;
            Font printFont = new Font("Courier New", 10);

            // Calculate the number of lines per page.
            linesPerPage = ev.MarginBounds.Height / printFont.GetHeight(ev.Graphics);

            // Print each line of the file.
            //Boolean prevblankline = false;
            while (count < linesPerPage && ((line = streamToPrint.ReadLine()) != null))
            {
                //dont print line if it is blank and the previous line was also blank

                Console.WriteLine(line);
                yPos = topMargin + (count * printFont.GetHeight(ev.Graphics));
                ev.Graphics.DrawString(line, printFont, Brushes.Black,
                leftMargin, yPos, new StringFormat());
                count++;
            }

            // If more lines exist, print another page.
            if (line != null)
                ev.HasMorePages = true;
            else
                ev.HasMorePages = false;
        }

        private void processCheckout()
        {
            Console.WriteLine("---StringReader---");
            StringReader sr = new StringReader(printedPage);
            Console.WriteLine("---Reset---");
            ckoSlip.reset(); //reset the values of the hold slip
            string line;
            int i = 1;
            Console.WriteLine("---Loop---");
            while ((line = sr.ReadLine()) != null)
            {
                if (line.Length > 0 && line != " ")
                {
                    Console.WriteLine("line {0}: '{1}'", i++, line);

                    if (ckoSlip.header == null) { ckoSlip.header = line; }
                    else if (ckoSlip.where == null) { ckoSlip.where = line; }
                    else if (ckoSlip.date == null) { ckoSlip.date = line; }
                    else if (ckoSlip.num == null) { ckoSlip.num = line; }
                    else if (ckoSlip.name == null) { ckoSlip.name = line; }
                    else if (line.StartsWith("Item:"))
                    {
                        CKOItem item = new CKOItem();
                        Console.WriteLine("---Begin Item Creation---");
                        item.num = line;
                        item.title = sr.ReadLine();
                        Console.WriteLine("line {0}: '{1}'", i++, item.title);
                        item.callno = sr.ReadLine();
                        Console.WriteLine("line {0}: '{1}'", i++, item.callno);
                        item.material = sr.ReadLine();
                        Console.WriteLine("line {0}: '{1}'", i++, item.material);
                        item.due = sr.ReadLine();
                        Console.WriteLine("line {0}: '{1}'", i++, item.due);

                        ckoSlip.addItem(item);
                        Console.WriteLine("---End Item Creation---");
                    }
                    else
                    {
                        Console.WriteLine("---Footer---");
                        ckoSlip.footer = line + sr.ReadToEnd();
                        Console.WriteLine("line {0}: '{1}'", i++, ckoSlip.footer);
                    }

                }
            }
            Console.WriteLine("########\n{0}\n##########", ckoSlip.toString());
        }

        private void processHold()
        {
            /* EXAMPLE
             * SALIS

    Pickup By: 11/07/2017

    3194700291672
    Game of thrones: The complete first season
    M

    U

    R

    P

    H

    Y

    ,

    A
    */
            StringReader sr = new StringReader(printedPage);

            holdSlip.reset(); //reset the values of the hold slip
            string line;
            //int i = 1;
            while ((line = sr.ReadLine()) != null)
            {
                if (line.Length > 0 && line != " ")
                {
                    //Console.WriteLine("line {0}: '{1}'", i++, line);

                    if (holdSlip.home == null) { holdSlip.home = line; }
                    else if (holdSlip.pickup == null) { holdSlip.pickup = line; }
                    else if (holdSlip.barcode == null) { holdSlip.barcode = line; }
                    else if (holdSlip.title == null) { holdSlip.title = line; }
                    else
                    {
                        holdSlip.name += Environment.NewLine + Environment.NewLine + line;
                    }

                }
            }
            Console.WriteLine("----\n{0}\n----", holdSlip.toString());
        }

        private void processTransit()
        {
            /*
             * To:
    Talbot - Easton
    IN-TRANSIT FOR HOLD 
    From:            SALIS
    Date:            Monday, Oct 30 2017 10:02AM
    Title:           Fire Twister
    Material Type:   DVD
    Call Number:     DVD F ACTIONADVENTU RE 0
    Item barcode:    30044001654013
    Assigned Branch: Somerset - Princess Anne Library
             */
            string slip = "";
            printedPage = Regex.Replace(printedPage, @"\r\n?|\n", " ");
            printedPage = Regex.Replace(printedPage, @"\t", " ");
            printedPage = Regex.Replace(printedPage, " {2,}", " ");
            //preserve the colon in the timestamp
            printedPage = Regex.Replace(printedPage, "([0-9][0-9])(:)([0-9][0-9])", "$1~$3");
            Console.WriteLine(printedPage);

            int[] indexes = new int[7];
            indexes[0] = printedPage.IndexOf("From:");
            indexes[1] = printedPage.IndexOf("Date:");
            indexes[2] = printedPage.IndexOf("Title:");
            indexes[3] = printedPage.IndexOf("Material Type:");
            indexes[4] = printedPage.IndexOf("Call Number:");
            indexes[5] = printedPage.IndexOf("Item barcode:");
            indexes[6] = printedPage.IndexOf("Assigned Branch:");
            int i = 1, maxChars = 36;
            string line = "";
            //get each line of the transit receipt
            for (i = 0; i < indexes.Length; i++)
            {

                //if at last index then grab the rest of the receipt
                if ((i + 1) == indexes.Length)
                {
                    line = printedPage.Substring(indexes[i]);
                }
                else
                {
                    line = printedPage.Substring(indexes[i], indexes[i + 1] - indexes[i]);
                }

                line = line.Replace(":", ": ");
                //Check to see if the line has too many chars to fit on one line
                //if so split it
                if (line.Length > maxChars)
                {
                    //line = line.Substring(0, maxChars) + Environment.NewLine + line.Substring(maxChars);
                }
                //Console.WriteLine(i + ": " + line);
                slip += line + Environment.NewLine;
            }

            //Create the transit header
            transitSlip.line1 = printedPage.Substring(0, 3); //To:

            int t = printedPage.IndexOf("IN-");
            transitSlip.line2 = printedPage.Substring(4, t - 5);
            transitSlip.line3 = printedPage.Substring(t, (indexes[0] - t));

            transitSlip.body = slip;

            //restore the colon in the timestamp
            transitSlip.body = Regex.Replace(transitSlip.body, "([0-9][0-9])(~)([0-9][0-9])", "$1:$3");

        }

        private void processPayment()
        {
            Console.WriteLine("---StringReader---");
            StringReader sr = new StringReader(printedPage);
            Console.WriteLine("---Reset---");
            paymentSlip.reset(); //reset the values of the hold slip
            string line;
            int i = 1;
            Console.WriteLine("---Loop---");
            bool headerDone = false;

            while ((line = sr.ReadLine()) != null)
            {
                if (!line.StartsWith("Title:") && !headerDone)
                {
                    paymentSlip.header += line + System.Environment.NewLine;
                    Console.WriteLine("line {0}: '{1}'", i++, line);
                }
                else if (!line.StartsWith("Total charges:"))
                {
                    headerDone = true;
                    paymentSlip.items += line + System.Environment.NewLine;
                    Console.WriteLine("line {0}: '{1}'", i++, line);
                }
                else
                {
                    Console.WriteLine("---Footer---");
                    paymentSlip.footer = line + System.Environment.NewLine + sr.ReadToEnd();
                    Console.WriteLine("line {0}: '{1}'", i++, paymentSlip.footer);
                }

            }
            Console.WriteLine("########\n{0}\n##########", paymentSlip.toString());

        }

        private void processCheckin()
        {
            Console.WriteLine("---StringReader---");
            StringReader sr = new StringReader(printedPage);
            Console.WriteLine("---Reset---");
            ckiSlip.reset(); //reset the values of the hold slip
            string line;
            int i = 1;
            Console.WriteLine("---Loop---");

            while ((line = sr.ReadLine()) != null)
            {
                if (!line.StartsWith("Barcode:"))
                {
                    ckiSlip.header += line + System.Environment.NewLine;
                    Console.WriteLine("line {0}: '{1}'", i++, line);
                }
                else
                {
                    Console.WriteLine("---Items---");
                    ckiSlip.items = line + System.Environment.NewLine + sr.ReadToEnd();
                    Console.WriteLine("line {0}: '{1}'", i++, ckiSlip.items);
                }

            }
            Console.WriteLine("########\n{0}\n##########", ckiSlip.toString());
        }

        private void pd_PrintPageString(object sender, PrintPageEventArgs e)
        {
            int charactersOnPage = 0;
            int linesPerPage = 0;

            Font printFont = new Font("Arial", 10);
            Font largeFont = new Font("Arial", 14, FontStyle.Bold);
            Font hugeFont = new Font("Arial", 20, FontStyle.Bold);

            /*Font printFont = new Font("Courier", 10);
            Font largeFont = new Font("Courier", 14, FontStyle.Bold);
            Font hugeFont = new Font("Courier", 20, FontStyle.Bold);
            */
            Brush sBrush = Brushes.Black;
            RectangleF rect1;
            Rectangle rect2;

            // Create a StringFormat object with the each line of text, and the block
            // of text centered on the page.
            StringFormat stringFormat = new StringFormat();
            stringFormat.Alignment = StringAlignment.Center;

            //stringFormat.LineAlignment = StringAlignment.Center;

            // Sets the value of charactersOnPage to the number of characters 
            // of printedPage that will fit within the bounds of the page.
            e.Graphics.MeasureString(printedPage, printFont,
                e.MarginBounds.Size, StringFormat.GenericTypographic,
                out charactersOnPage, out linesPerPage);

            switch (pageType)
            {
                case TRANSIT:

                    e.Graphics.DrawString(transitSlip.line1 + Environment.NewLine, printFont, sBrush, new RectangleF(10, 10, paperSize.Width - 20, 20), stringFormat);
                    float h = Convert.ToSingle(Math.Ceiling(transitSlip.line2.Length / 25.0)) * 25;
                    //Console.WriteLine("Height: " + h);
                    rect1 = new RectangleF(10, 30, paperSize.Width - 20, h);
                    //e.Graphics.FillRectangle(new SolidBrush(Color.LightGray), rect1);
                    e.Graphics.DrawString(transitSlip.line2 + Environment.NewLine, largeFont, sBrush, rect1, stringFormat);
                    e.Graphics.DrawString(transitSlip.line3 + Environment.NewLine, printFont, sBrush, new RectangleF(10, 35 + h, paperSize.Width - 20, 20), stringFormat);

                    //Draw the transit body
                    float bh = Convert.ToSingle(Math.Ceiling(transitSlip.body.Length / 35.0)) * 20;
                    //Console.WriteLine("Body Height: " + Convert.ToSingle(Math.Ceiling(transitSlip.body.Length / 35.0)) + " | " + bh);
                    RectangleF bodyRect = new RectangleF(10, 70 + h, paperSize.Width - 20, bh);
                    //e.Graphics.FillRectangle(new SolidBrush(Color.Blue), bodyRect);
                    e.Graphics.DrawString(transitSlip.body, printFont, sBrush, bodyRect);

                    //Only print first page
                    e.HasMorePages = false;

                    break;
                case HOLD:
                    //Draw each line of the hold slip
                    e.Graphics.DrawString(holdSlip.home, printFont, sBrush, 550, 50, stringFormat);
                    e.Graphics.DrawString(holdSlip.pickup, largeFont, sBrush, 550, 80, stringFormat);
                    e.Graphics.DrawString(holdSlip.barcode, printFont, sBrush, 550, 110, stringFormat);
                    e.Graphics.DrawString(holdSlip.title, printFont, sBrush, 550, 130, stringFormat);
                    e.Graphics.DrawString(holdSlip.name, hugeFont, sBrush, 550, 150, stringFormat);

                    //Only print the first page
                    e.HasMorePages = false;

                    break;
                case CKO:
                    Console.WriteLine("---Creating CKO Slip---");
                    //string slip = ckoSlip.toString();
                    //Draw each line of the checkout slip

                    /*e.Graphics.DrawString(ckoSlip.header, printFont, sBrush, 20, 20);
                    e.Graphics.DrawString(ckoSlip.where, printFont, sBrush, 20, 40);
                    e.Graphics.DrawString(ckoSlip.date, printFont, sBrush, 20, 60);
                    e.Graphics.DrawString(ckoSlip.num, printFont, sBrush, 20, 80);
                    e.Graphics.DrawString(ckoSlip.name, printFont, sBrush, 20, 100);
                    */
                    /*int y = 120;
                    string items = "";
                    foreach(CKOItem i in ckoSlip.getItems())
                    {
                        items += i.toGenericString() + Environment.NewLine + Environment.NewLine;
                        y += 90;
                    }
                    *
                    int count = 0;
                    foreach (char c in slip)
                    {
                        if (c == '/') count++;
                    }
                    */

                    rect2 = new Rectangle(10, 10, paperSize.Width - 20, paperSize.Height - 10);
                    e.Graphics.DrawString(printedPage, printFont, sBrush, rect2);// rect1);
                    e.Graphics.MeasureString(printedPage, printFont,
                        rect2.Size, StringFormat.GenericTypographic,
                        out charactersOnPage, out linesPerPage);
                    //e.Graphics.DrawString(items, printFont, sBrush, 20, y);

                    //Remove the portion of the string that has been printed.
                    Console.WriteLine("Chars on page: " + charactersOnPage);
                    printedPage = printedPage.Substring(charactersOnPage).Trim();

                    //Check to see if more pages are to be printed.
                    e.HasMorePages = (printedPage.Length > 0);
                    break;
                case PAYMENT:
                    Console.WriteLine("---Creating PAYMENT Slip---");
                    rect2 = new Rectangle(10, 10, paperSize.Width - 20, paperSize.Height - 10);
                    e.Graphics.DrawString(printedPage, printFont, sBrush, rect2);// rect1);
                    e.Graphics.MeasureString(printedPage, printFont,
                        rect2.Size, StringFormat.GenericTypographic,
                        out charactersOnPage, out linesPerPage);
                    //e.Graphics.DrawString(items, printFont, sBrush, 20, y);

                    //Remove the portion of the string that has been printed.
                    Console.WriteLine("Chars on page: " + charactersOnPage);
                    printedPage = printedPage.Substring(charactersOnPage).Trim();

                    //Check to see if more pages are to be printed.
                    e.HasMorePages = (printedPage.Length > 0);
                    break;
                case CKI:
                    Console.WriteLine("---Creating CHECK-IN Slip---");
                    rect2 = new Rectangle(10, 10, paperSize.Width - 20, paperSize.Height - 10);
                    e.Graphics.DrawString(printedPage, printFont, sBrush, rect2);// rect1);
                    e.Graphics.MeasureString(printedPage, printFont,
                        rect2.Size, StringFormat.GenericTypographic,
                        out charactersOnPage, out linesPerPage);
                    //e.Graphics.DrawString(items, printFont, sBrush, 20, y);

                    //Remove the portion of the string that has been printed.
                    Console.WriteLine("Chars on page: " + charactersOnPage);
                    printedPage = printedPage.Substring(charactersOnPage).Trim();

                    //Check to see if more pages are to be printed.
                    e.HasMorePages = (printedPage.Length > 0);
                    break;
                case NONE:
                    Console.WriteLine("---Creating NO MATCH Slip---");
                    rect2 = new Rectangle(10, 10, 1100 - 30, 8500 - 30);
                    e.Graphics.DrawString(printedPage, printFont, sBrush, rect2);
                    e.Graphics.MeasureString(printedPage, printFont,
                        rect2.Size, StringFormat.GenericTypographic,
                        out charactersOnPage, out linesPerPage);
                    //e.Graphics.DrawString(items, printFont, sBrush, 20, y);

                    //Remove the portion of the string that has been printed.
                    Console.WriteLine("Chars on page: " + charactersOnPage);
                    printedPage = printedPage.Substring(charactersOnPage).Trim();

                    //Check to see if more pages are to be printed.
                    e.HasMorePages = (printedPage.Length > 0);
                    break;
                default:
                    e.Graphics.DrawString(printedPage, printFont, sBrush, 10, 110);
                    e.HasMorePages = false;
                    break;
            }

        }
    }
}
